<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use App\Models\Admin;
use App\Models\Plan;
use App\Models\User;
use App\Models\UserPlan;
use App\Models\Deposit;
use App\Models\Withdrawal;
use App\Models\TpTransaction;
use App\Models\Testimony;
use App\Models\Agent;
use App\Models\Faq;
use App\Models\Content;
use App\Models\Image;
use App\Models\Setting;
use Mail;

class AdminController extends Controller
{
    //
    public function __construct()
    {
    	$this->path = 'admin/';
    }

    //
    public function login()
    {
    	return view($this->path . 'login');
    }

    //
    public function login_post(Request $request)
    {

        try{
            $admin = Admin::where("email", $request->email)->first();

            if(!$admin){
                $body = 'email or username does not exist';
                toast('error', $body);
                return redirect()->back()->withInput();
            }

            if(!\Hash::check($request->password, $admin->password)){
                $body = 'Incorrect password, please try again';
                toast('error', $body);
                return redirect()->back()->withInput();
            }

            \Auth::guard('admin')->login($admin);

            toast("success", "signin successful");
            return redirect()->route('admin.dashboard');
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    //
    public function logout()
    {
        \Auth::guard('admin')->logout();
        return redirect()->route('admin.login');
    }

    //
    public function dashboard()
    {
        return view($this->path . 'dashboard');
    }

    //
    public function account_update(Request $request)
    {

        try{
            $admin = Admin::where("id", admin()->id)->first();

            $admin->update([
                'firstName' => $request->firstName ?? $admin->firstName,
                'lastName' => $request->lastName ?? $admin->lastName,
                'phone' => $request->phone ?? $admin->phone,
                'enable_2fa' => $request->enable_2fa ?? $admin->enable_2fa,
            ]);

            toast("success", "admin updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function change_password(Request $request)
    {

        try{
            $admin = Admin::where("id", admin()->id)->first();

            if(\Hash::check($request->old_password, $admin->password)){
                if($request->new_password != $request->confirm_new_password){
                    toast("error", "password mismatch");
                    return redirect()->back();
                }

                $admin->update([
                    'password' => $request->new_password ? bcrypt($request->new_password) : $admin->password
                ]);

                toast("success", "admin updated successfully");
                return redirect()->back();
            }

            toast("error", "wrong old password");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function plan()
    {
        return view($this->path . 'plan')->with([
            'plans' => Plan::all()
        ]);
    }

    //
    public function plan_add(Request $request)
    {
        $request->validate([
            'name' => 'required',
            'price' => 'required',
            'min_price' => 'required',
            'max_price' => 'required',
            'minr' => 'required',
            'maxr' => 'required',
            'gift' => 'required',
            'increment_interval' => 'required',
            'increment_type' => 'required',
            'increment_amount' => 'required',
            'expiration' => 'required',
        ]);

        try{
            Plan::create([
                'name' => $request->name,
                'price' => $request->price,
                'min_price' => $request->min_price,
                'max_price' => $request->max_price,
                'minr' => $request->minr,
                'maxr' => $request->maxr,
                'gift' => $request->gift,
                'increment_interval' => $request->increment_interval,
                'increment_type' => $request->increment_type,
                'increment_amount' => $request->increment_amount,
                'expiration' => $request->expiration,
            ]);

            toast("success", "plan added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function plan_edit(Request $request)
    {

        try{
            $plan = Plan::where("id", $request->id)->first();

            if(!$plan){
                toast("error", "plan not found");
                return redirect()->back();
            }

            $plan->update([
                'name' => $request->name ?? $plan->name,
                'price' => $request->price ?? $plan->price,
                'min_price' => $request->min_price ?? $plan->min_price,
                'max_price' => $request->max_price ?? $plan->max_price,
                'minr' => $request->minr ?? $plan->minr,
                'maxr' => $request->maxr ?? $plan->maxr,
                'gift' => $request->gift ?? $plan->gift,
                'increment_interval' => $request->increment_interval ?? $plan->increment_interval,
                'increment_type' => $request->increment_type ?? $plan->increment_type,
                'increment_amount' => $request->increment_amount ?? $plan->increment_amount,
                'expiration' => $request->expiration ?? $plan->expiration,
            ]);

            toast("success", "plan updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function plan_delete(Request $request)
    {

        try{
            $plan = Plan::where("id", $request->id)->first();

            if(!$plan){
                toast("error", "plan not found");
                return redirect()->back();
            }

            $plan->delete();

            toast("success", "plan deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function users(Request $request)
    {
        return view($this->path . 'users')->with([
            'users' => User::orderBy("id", "desc")->where("name", "like", "%" . $request->search . "%")->paginate(10),
            'plans' => Plan::all(),
            'search' => $request->search
        ]);
    }

    //
    public function user_add(Request $request)
    {
        $request->validate([
            'email' => 'required|unique:users,email',
            'name' => 'required',
            'phone' => 'required',
            'password' => 'required|same:confirm_password',
        ]);

        try{
            User::create([
                'name' => $request->name,
                'email' => $request->email,
                'phone' => $request->phone,
                'password' => bcrypt($request->password),
                'signup_bonus' => 'received',
                'account_verify' => 'Verified',
                'status' => 'active',
            ]);

            toast("success", "user added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_plans(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            return view($this->path . 'user_plans')->with([
                'user' => $user,
                'user_plans' => UserPlan::leftJoin("plans", "plans.id", "=", "user_plans.plan")
                                    ->select([
                                        'plans.name as name',
                                        'user_plans.id as id',
                                        'user_plans.amount as amount',
                                        'user_plans.active as active',
                                        'user_plans.inv_duration as inv_duration',
                                        'user_plans.activated_at as activated_at',
                                        'user_plans.expire_date as expire_date',
                                    ])
                                    ->orderBy("id", "desc")
                                    ->where("user", $request->id)
                                    ->get(),
            ]);
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_wallet(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            return view($this->path . 'user_wallet')->with([
                'user' => $user,
            ]);
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_block(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'status' => 'blocked'
            ]);

            toast("success", "user blocked successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_activate(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'status' => 'active'
            ]);

            toast("success", "user activated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_turn_off_trade(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'trade_mode' => 'off'
            ]);

            toast("success", "trade mode modified successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_turn_on_trade(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'trade_mode' => 'on'
            ]);

            toast("success", "trade mode modified successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_credit_debit(Request $request)
    {
        $request->validate([
            'amount' => 'required',
        ]);

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            if($request->column == "Deposit"){
                Deposit::create([
                    'user' => $request->id,
                    'amount' => $request->type == 'Debit' ? ($request->amount * -1) : $request->amount,
                    'payment_mode' => 'Express Deposit',
                    'status' => 'Processed',
                ]);

                //increment or decrement user balance
                $user->update([
                    'account_bal' => $request->type == 'Debit' ? ($user->account_bal - $request->amount) : ($user->account_bal + $request->amount)
                ]);

                //add transaction
                TpTransaction::create([
                    'user' => $user->id,
                    'amount' => $request->type == 'Debit' ? ($request->amount * -1) : $request->amount,
                    'type' => 'Express Deposit'
                ]);

                toast("success", "deposit made successfully");
                return redirect()->back();
            }
            elseif($request->column == "Bonus"){
                $user->update([
                    'bonus' => $request->type == 'Debit' ? ($user->bonus - $request->amount) : ($user->bonus + $request->amount),
                    'account_bal' => $request->type == 'Debit' ? ($user->account_bal - $request->amount) : ($user->account_bal + $request->amount)
                ]);

                //add transaction
                TpTransaction::create([
                    'user' => $user->id,
                    'amount' => $request->type == 'Debit' ? ($request->amount * -1) : $request->amount,
                    'type' => 'Bonus'
                ]);
            }
            elseif($request->column == "Ref_Bonus"){
                $user->update([
                    'ref_bonus' => $request->type == 'Debit' ? ($user->ref_bonus - $request->amount) : ($user->ref_bonus + $request->amount),
                    'account_bal' => $request->type == 'Debit' ? ($user->account_bal - $request->amount) : ($user->account_bal + $request->amount)
                ]);

                //add transaction
                TpTransaction::create([
                    'user' => $user->id,
                    'amount' => $request->type == 'Debit' ? ($request->amount * -1) : $request->amount,
                    'type' => 'Ref Bonus'
                ]);
            }
            else{
                $user->update([
                    'roi' => $request->type == 'Debit' ? ($user->roi - $request->amount) : ($user->roi + $request->amount),
                    'account_bal' => $request->type == 'Debit' ? ($user->account_bal - $request->amount) : ($user->account_bal + $request->amount)
                ]);

                //add transaction
                TpTransaction::create([
                    'user' => $user->id,
                    'amount' => $request->type == 'Debit' ? ($request->amount * -1) : $request->amount,
                    'type' => 'ROI'
                ]);
            }

            $user->update([
                'account_bal' => ($user->account_bal + $deposit->amount)
            ]);

            $data = array(
                'name' => $user->name,
                'body' => 'Your account has been ' . $request->type . 'ed with the sum of of $'. number_format($request->amount, 2, '.', ',') . ', please login to your account to view.',
            );

            $mail = $user->email;

            $type = $request->type;
            
            Mail::send('mails.notify3', $data, function($message) use ($mail, $type) {
                $message->to($mail)->subject( $type. ' Notification');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            //notify use about the deposit
            $message = "A sum of $" . number_format($request->amount, 2, '.', ',')  . " has been ". $request->type ."ed to your account";
            notify($message, $user->id);

            toast("success", "user update successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_reset_password(Request $request)
    {
        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'password' => bcrypt('user01236')
            ]);

            toast("success", "password modified successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_clear_account(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'account_bal' => 0,
                'bonus' => 0,
                'ref_bonus' => 0,
                'roi' => 0,
            ]);

            //clear users deposit withdrawals and tp_transactions
            Deposit::where("user", $request->id)->delete();
            Withdrawal::where("user", $request->id)->delete();
            TpTransaction::where("user", $request->id)->delete();

            toast("success", "user account cleared successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_add_trading_history(Request $request)
    {
        $request->validate([
            'amount' => 'required',
            'plan' => 'required',
            'type' => 'required',
        ]);

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            TpTransaction::create([
                'plan' => $request->plan,
                'user' => $request->id,
                'amount' => $request->amount,
                'type' => $request->type,
            ]);

            //now we update user balances to reflect the added trading history
            $user->update([
                'account_bal' => ($user->account_bal + $request->amount),
                'roi' => $request->type == "ROI" ? ($user->roi + $request->amount) : $user->roi,
                'bonus' => $request->type == "Bonus" ? ($user->bonus + $request->amount) : $user->bonus,
            ]);

            toast("success", "trading history added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_delete(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->delete();

            toast("success", "user deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_edit(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            if($user->email != $request->email){
                $request->validate([
                    'email' => 'unique:users,email'
                ]);
            }

            if($user->ref_link != $request->ref_link){
                $request->validate([
                    'ref_link' => 'unique:users,ref_link'
                ]);
            }

            $user->update([
                'name' => $request->name ?? $user->name,
                'email' => $request->email ?? $user->email,
                'phone' => $request->phone ?? $user->phone,
                'ref_link' => $request->ref_link ?? $user->ref_link,
            ]);

            toast("success", "user updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_mail(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if($user){
                //send to single user and return
                $data = array(
                    'subject' => $request->subject,
                    'body' => $request->body,
                );

                $mail = $user->email;
                
                Mail::send('mails.notify2', $data, function($message) use ($mail) {
                    $message->to($mail)->subject('Notification');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                $data2 = array(
                    'admin' => admin()->firstName . ' ' . admin()->lastName,
                    'user' => $user->name,
                    'body' => $request->body,
                );

                $admin_mail = admin_email();
                
                Mail::send('mails.notify5', $data2, function($message) use ($admin_mail) {
                    $message->to($mail)->subject('Admin Activity Update');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                toast("success", "mail sent successfully");
                return redirect()->back();
            }

            //send to all users

            $data = array(
                'subject' => $request->subject,
                'body' => $request->body,
            );

            $mails = json_decode('[]', TRUE);

            $users = User::orderBy("id")->get();

            foreach($users as $user){
                $user->email ? $mails[] = $user->email : null;
            }
            
            Mail::send('mails.notify2', $data, function($message) use ($mails) {
                $message->bcc($mails);
                $message->subject('Notification');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            $data2 = array(
                'admin' => admin()->firstName . ' ' . admin()->lastName,
                'user' => "all users",
                'body' => $request->body,
            );

            $admin_mail = admin_email();
            
            Mail::send('mails.notify5', $data2, function($message) use ($admin_mail) {
                $message->to($mail)->subject('Admin Activity Update');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            toast("success", "mail sent successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_kyc(Request $request)
    {
        return view($this->path . 'user_kyc')->with([
            'users' => User::orderBy("id", "desc")
                        ->where("id_card", "!=", NULL)
                        ->where("passport", "!=", NULL)
                        ->where("account_verify", "!=", "Verified")
                        ->where("name", "like", "%" . $request->search . "%")
                        ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function user_kyc_accept(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            $user->update([
                'account_verify' => 'Verified',
            ]);

            $message = "Your KYC details has been accepted and your account verified";
            notify($message, $user->id);

            toast("success", "user verified successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_kyc_reject(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            //delete the id and passport
            Storage::disk("public")->delete($user->id_card);
            Storage::disk("public")->delete($user->passport);

            $user->update([
                'account_verify' => NULL,
                'id_card' => NULL,
                'passport' => NULL,
            ]);

            toast("success", "user KYC rejected successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function user_get_access(Request $request)
    {

        try{
            $user = User::where("id", $request->id)->first();

            if(!$user){
                toast("error", "user not found");
                return redirect()->back();
            }

            \Auth::guard("web")->login($user);

            toast("success", "user login successfully");
            return redirect()->route('user.dashboard');
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function deposits(Request $request)
    {
        return view($this->path . 'deposits')->with([
            'deposits' => Deposit::leftJoin("users", "users.id", "=", "deposits.user")
                        ->leftJoin("plans", "plans.id", "=", "deposits.plan")
                        ->select([
                            'users.name as name',
                            'users.email as email',
                            'plans.name as plan',
                            'deposits.id as id',
                            'deposits.amount as amount',
                            'deposits.payment_mode as payment_mode',
                            'deposits.status as status',
                            'deposits.proof as proof',
                            'deposits.created_at as created_at'
                        ])
                        ->orderBy("deposits.id", "desc")
                        ->where("users.name", "like", "%" . $request->search . "%")
                        ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function deposit_delete(Request $request)
    {

        try{
            $deposit = Deposit::where("id", $request->id)->first();

            if(!$deposit){
                toast("error", "deposit not found");
                return redirect()->back();
            }

            if($deposit->status == "Processed"){
                toast("error", "this deposit has been processed");
                return redirect()->back();
            }

            //delete the uploaded proof here
            Storage::disk("public")->delete($deposit->proof);

            $deposit->delete();

            toast("success", "deposit deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function deposit_process(Request $request)
    {

        try{
            $deposit = Deposit::where("id", $request->id)->first();

            if(!$deposit){
                toast("error", "deposit not found");
                return redirect()->back();
            }

            if($deposit->status == "Processed"){
                toast("info", "this deposit has been processed");
                return redirect()->back();
            }

            $deposit->update([
                'status' => 'Processed',
            ]);

            //add the transaction
            TpTransaction::create([
                'user' => $deposit->user,
                'amount' => $deposit->amount,
                'type' => 'Deposit'
            ]);

            //increase user balance
            $user = User::where("id", $deposit->user)->first();

            if($user){
                $user->update([
                    'account_bal' => ($user->account_bal + $deposit->amount)
                ]);

                $data = array(
                    'name' => $user->name,
                    'body' => 'Your deposit request of $'. number_format($deposit->amount, 2, '.', ',') . ' has been confirmed, please login to your account to view.',
                );

                $mail = $user->email;
                
                Mail::send('mails.notify3', $data, function($message) use ($mail) {
                    $message->to($mail)->subject('Deposit Confirmation');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                //notify use about the deposit
                $message = "Your deposit request of $" . number_format($deposit->amount, 2, '.', ',')  . " has been confirmed and your account credited accordingly";
                notify($message, $user->id);
            }

            toast("success", "deposit processed successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function withdrawals(Request $request)
    {
        return view($this->path . 'withdrawals')->with([
            'withdrawals' => Withdrawal::leftJoin("users", "users.id", "=", "withdrawals.user")
                        ->select([
                            'users.name as name',
                            'users.email as email',
                            'users.btc_address as btc_address',
                            'users.eth_address as eth_address',
                            'users.ltc_address as ltc_address',
                            'users.ltc_address as dodge_address',
                            'users.ltc_address as bnb_address',
                            'withdrawals.id as id',
                            'withdrawals.amount as amount',
                            'withdrawals.to_deduct as to_deduct',
                            'withdrawals.payment_mode as payment_mode',
                            'withdrawals.status as status',
                            'withdrawals.created_at as created_at'
                        ])
                        ->orderBy("withdrawals.id", "desc")
                        ->where("users.name", "like", "%" . $request->search . "%")
                        ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function withdrawal_reject(Request $request)
    {

        try{
            $withdrawal = Withdrawal::where("id", $request->id)->first();

            if(!$withdrawal){
                toast("error", "withdrawal not found");
                return redirect()->back();
            }

            if($withdrawal->status == "Processed"){
                toast("error", "this withdrawal cannot be rejected because it has already been processed");
                return redirect()->back();
            }

            //return the users balance
            $user = User::where("id", $withdrawal->user)->first();
            if($user){
                $user->update([
                    'account_bal' => $user->account_bal + $withdrawal->amount
                ]);
            }

            $withdrawal->delete();

            toast("success", "withdrawal rejected successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function withdrawal_process(Request $request)
    {

        try{
            $withdrawal = Withdrawal::where("id", $request->id)->first();

            if(!$withdrawal){
                toast("error", "withdrawal not found");
                return redirect()->back();
            }

            if($withdrawal->status == "Processed"){
                toast("info", "this withdrawal cannot be rejected because it has already been processed");
                return redirect()->back();
            }

            $withdrawal->update([
                'status' => 'Processed',
            ]);

            //add it to transaction
            TpTransaction::create([
                'user' => $withdrawal->user,
                'amount' => $withdrawal->amount,
                'type' => 'Withdrawal'
            ]);

            //send confirmation email
            $user = User::where("id", $withdrawal->user)->first();

            if($user){
                $data = array(
                    'name' => $user->name,
                    'body' => 'Your withdrawal request of $'. number_format(($withdrawal->amount), 2, '.', ',') . ' has been successfully sent to your '. $withdrawal->payment_mode .' address, please login to your account to view.',
                );

                $mail = $user->email;
                
                Mail::send('mails.notify3', $data, function($message) use ($mail) {
                    $message->to($mail)->subject('Withdrawal Confirmation');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                //notify user
                $message = 'Your withdrawal request of $'. number_format(($withdrawal->amount), 2, '.', ',') . ' has been successfully sent to your '. $withdrawal->payment_mode .' address';
                notify($message, $user->id);
            }

            toast("success", "withdrawal processed successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function agents(Request $request)
    {
        return view($this->path . 'agents')->with([
            'users' => User::orderBy("name")->get(),
            'agents' => Agent::leftJoin("users", "users.id", "=", "agents.agent")
                        ->select([
                            'users.name as name',
                            'users.id as user_id',
                            'agents.id as id',
                            'agents.total_refered as total_refered'
                        ])
                        ->orderBy("agents.id", "desc")
                        ->where("users.name", "like", "%" . $request->search . "%")
                        ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function agent_add(Request $request)
    {
        $request->validate([
            'agent' => 'required|unique:agents,agent',
            'total_refered' => 'required',
        ]);

        try{
            Agent::create([
                'agent' => $request->agent,
                'total_refered' => $request->total_refered
            ]);

            toast("success", "agent added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function agent_clients(Request $request)
    {
        return view($this->path . 'agent_clients')->with([
            'user' => User::where("id", $request->id)->first(),
            'clients' => User::leftJoin("plans", "plans.id", "=", "users.plan")
                        ->select([
                            'users.id as id',
                            'users.name as name',
                            'users.roi as roi',
                            'users.status as status',
                            'plans.name as plan',
                        ])
                        ->orderBy("users.name")
                        ->where("users.ref_by", $request->id)
                        ->paginate(10)
        ]);
    }

    //
    public function agent_delete(Request $request)
    {

        try{
            $agent = Agent::where("id", $request->id)->first();

            if(!$agent){
                toast("error", "agent not found");
                return redirect()->back();
            }

            $agent->delete();

            toast("success", "agent deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrators(Request $request)
    {
        return view($this->path . 'administrators')->with([
            'administrators' => Admin::orderBy("id", "desc")
                                ->where("firstName", "like", "%" . $request->search . "%")
                                ->orWhere("lastName", "like", "%" . $request->search . "%")
                                ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function administrator_add(Request $request)
    {
        $request->validate([
            'email' => 'required|unique:admins,email',
            'firstName' => 'required',
            'lastName' => 'required',
            'phone' => 'required',
            'type' => 'required',
            'password' => 'required|same:confirm_password',
        ]);

        try{

            if(admin()->type != "Super Admin"){
                toast("error", "Only super admin can add new administrator.");
                return redirect()->back();
            }

            if($request->type == "Super Admin"){
                toast("error", "you cannot add a super admin.");
                return redirect()->back();
            }

            Admin::create([
                'firstName' => $request->firstName,
                'lastName' => $request->lastName,
                'email' => $request->email,
                'phone' => $request->phone,
                'type' => $request->type,
                'password' => bcrypt($request->password),
                'enable_2fa' => 'disabled',
                'acnt_type_active' => 'active',
                'status' => 'active',
            ]);

            toast("success", "administrator added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrator_block(Request $request)
    {
        try{
            $administrator = Admin::where("id", $request->id)->first();

            if(!$administrator){
                toast("error", "administrator not found");
                return redirect()->back();
            }

            if($administrator->type == "Super Admin"){
                toast("error", "super admin accounts cannot be blocked.");
                return redirect()->back();
            }

            $administrator->update([
                'status' => 'blocked'
            ]);

            toast("success", "administrator blocked successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrator_activate(Request $request)
    {
        try{
            $administrator = Admin::where("id", $request->id)->first();

            if(!$administrator){
                toast("error", "administrator not found");
                return redirect()->back();
            }

            $administrator->update([
                'status' => 'active'
            ]);

            toast("success", "administrator activated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrator_reset_password(Request $request)
    {
        try{
            $administrator = Admin::where("id", $request->id)->first();

            if(!$administrator){
                toast("error", "administrator not found");
                return redirect()->back();
            }

            $administrator->update([
                'password' => bcrypt('admin01236')
            ]);

            toast("success", "password modified successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrator_delete(Request $request)
    {

        try{
            $administrator = Admin::where("id", $request->id)->first();

            if(!$administrator){
                toast("error", "administrator not found");
                return redirect()->back();
            }

            if($administrator->type == "Super Admin"){
                toast("error", "super admin accounts cannot be deleted.");
                return redirect()->back();
            }

            $administrator->delete();

            toast("success", "administrator deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrator_edit(Request $request)
    {

        try{
            $administrator = Admin::where("id", $request->id)->first();

            if(!$administrator){
                toast("error", "administrator not found");
                return redirect()->back();
            }

            if($administrator->type == "Super Admin" && admin()->type == "Admin"){
                toast("error", "you cannot modify super admin account.");
                return redirect()->back();
            }

            if($administrator->email != $request->email){
                $request->validate([
                    'email' => 'unique:admins,email'
                ]);
            }

            $administrator->update([
                'firstName' => $request->firstName ?? $administrator->firstName,
                'lastName' => $request->lastName ?? $administrator->lastName,
                'email' => $request->email ?? $administrator->email,
                'phone' => $request->phone ?? $administrator->phone,
                'type' => $request->type ?? $administrator->type,
            ]);

            toast("success", "administrator updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function administrator_mail(Request $request)
    {

        try{
            $administrator = Admin::where("id", $request->id)->first();

            //send to single user and return

            $data = array(
                'message_body' => $request->message
            );

            $mail = $administrator->email;
            
            /*Mail::send('mails.bulk', $data, function($message) use ($mail) {
                $message->to($mail);
                $message->subject('Notification');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });*/

            toast("success", "mail sent successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function faqs(Request $request)
    {
        return view($this->path . 'faqs')->with([
            'faqs' => Faq::orderBy("id", "desc")
                                ->where("question", "like", "%" . $request->search . "%")
                                ->orWhere("answer", "like", "%" . $request->search . "%")
                                ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function faq_add(Request $request)
    {
        $request->validate([
            'ref_key' => 'required',
            'question' => 'required',
            'answer' => 'required',
        ]);

        try{
            Faq::create([
                'ref_key' => $request->ref_key,
                'question' => $request->question,
                'answer' => $request->answer,
            ]);

            toast("success", "faq added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function faq_delete(Request $request)
    {

        try{
            $faq = Faq::where("id", $request->id)->first();

            if(!$faq){
                toast("error", "faq not found");
                return redirect()->back();
            }

            $faq->delete();

            toast("success", "faq deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function faq_edit(Request $request)
    {

        try{
            $faq = Faq::where("id", $request->id)->first();

            if(!$faq){
                toast("error", "faq not found");
                return redirect()->back();
            }

            $faq->update([
                'ref_key' => $request->ref_key ?? $faq->ref_key,
                'question' => $request->question ?? $faq->question,
                'answer' => $request->answer ?? $faq->answer,
            ]);

            toast("success", "faq updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function testimonials(Request $request)
    {
        return view($this->path . 'testimonials')->with([
            'testimonials' => Testimony::orderBy("id", "desc")
                                ->where("name", "like", "%" . $request->search . "%")
                                ->orWhere("what_is_said", "like", "%" . $request->search . "%")
                                ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function testimonial_add(Request $request)
    {
        $request->validate([
            'ref_key' => 'required',
            'position' => 'required',
            'name' => 'required',
            'what_is_said' => 'required',
            'picture' => 'required|max:1024|mimes:jpg,jpeg,png',
        ]);

        try{
            $picture = Storage::disk("public")->put("testimonies", $request->file("picture"));

            Testimony::create([
                'ref_key' => $request->ref_key,
                'position' => $request->position,
                'name' => $request->name,
                'what_is_said' => $request->what_is_said,
                'picture' => $picture,
            ]);

            toast("success", "testimony added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function testimonial_delete(Request $request)
    {

        try{
            $testimony = Testimony::where("id", $request->id)->first();

            if(!$testimony){
                toast("error", "testimony not found");
                return redirect()->back();
            }
            
            //delete image
            Storage::disk("public")->delete($testimony->picture);

            $testimony->delete();

            toast("success", "testimony deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function testimonial_edit(Request $request)
    {

        try{
            $testimony = Testimony::where("id", $request->id)->first();

            if(!$testimony){
                toast("error", "testimony not found");
                return redirect()->back();
            }

            if($testimony->picture){
                $picture = $request->file('picture') ? Storage::disk('public')->putFileAs(NULL, $request->file('picture'), $testimony->picture) : $testimony->picture;
            }
            else{
                $picture = $request->file('picture') ? Storage::disk('public')->put('testimonies', $request->file('picture')) : $testimony->picture;
            }

            $testimony->update([
                'ref_key' => $request->ref_key ?? $testimony->ref_key,
                'position' => $request->position ?? $testimony->position,
                'name' => $request->name ?? $testimony->name,
                'what_is_said' => $request->what_is_said ?? $testimony->what_is_said,
                'picture' => $picture,
            ]);

            toast("success", "testimony updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function contents(Request $request)
    {
        return view($this->path . 'contents')->with([
            'contents' => Content::orderBy("id", "desc")
                                ->where("title", "like", "%" . $request->search . "%")
                                ->orWhere("description", "like", "%" . $request->search . "%")
                                ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function content_add(Request $request)
    {
        $request->validate([
            'ref_key' => 'required',
            'title' => 'required',
            'description' => 'required',
        ]);

        try{
            Content::create([
                'ref_key' => $request->ref_key,
                'title' => $request->title,
                'description' => $request->description,
            ]);

            toast("success", "content added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function content_delete(Request $request)
    {

        try{
            $content = Content::where("id", $request->id)->first();

            if(!$content){
                toast("error", "content not found");
                return redirect()->back();
            }

            $content->delete();

            toast("success", "content deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function content_edit(Request $request)
    {

        try{
            $content = Content::where("id", $request->id)->first();

            if(!$content){
                toast("error", "content not found");
                return redirect()->back();
            }

            $content->update([
                'ref_key' => $request->ref_key ?? $content->ref_key,
                'title' => $request->title ?? $content->title,
                'description' => $request->description ?? $content->description,
            ]);

            toast("success", "content updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function images(Request $request)
    {
        return view($this->path . 'images')->with([
            'images' => Image::orderBy("id", "desc")
                                ->where("title", "like", "%" . $request->search . "%")
                                ->orWhere("description", "like", "%" . $request->search . "%")
                                ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function image_add(Request $request)
    {
        $request->validate([
            'ref_key' => 'required',
            'title' => 'required',
            'description' => 'required',
            'img_path' => 'required|max:1024|mimes:jpg,jpeg,png',
        ]);

        try{
            $img_path = Storage::disk("public")->put("images", $request->file("img_path"));

            Image::create([
                'ref_key' => $request->ref_key,
                'title' => $request->title,
                'description' => $request->description,
                'img_path' => $img_path,
            ]);

            toast("success", "image added successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function image_delete(Request $request)
    {

        try{
            $image = Image::where("id", $request->id)->first();

            if(!$image){
                toast("error", "image not found");
                return redirect()->back();
            }

            //delete image
            Storage::disk("public")->delete($image->img_path);

            $image->delete();

            toast("success", "image deleted successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function image_edit(Request $request)
    {

        try{
            $image = Image::where("id", $request->id)->first();

            if(!$image){
                toast("error", "image not found");
                return redirect()->back();
            }

            if($image->img_path){
                $img_path = $request->file('img_path') ? Storage::disk('public')->putFileAs(NULL, $request->file('img_path'), $image->img_path) : $image->img_path;
            }
            else{
                $img_path = $request->file('img_path') ? Storage::disk('public')->put('images', $request->file('img_path')) : $image->img_path;
            }

            $image->update([
                'ref_key' => $request->ref_key ?? $image->ref_key,
                'title' => $request->title ?? $image->title,
                'description' => $request->description ?? $image->description,
                'img_path' => $img_path,
            ]);

            toast("success", "image updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function settings(Request $request)
    {
        return view($this->path . 'settings')->with([
            'setting' => Setting::first(),
        ]);
    }

    //
    public function setting_edit(Request $request)
    {

        try{
            $setting = Setting::first();

            if($setting->logo){
                $logo = $request->file('logo') ? Storage::disk('public')->putFileAs(NULL, $request->file('logo'), $setting->logo) : $setting->logo;
            }
            else{
                $logo = $request->file('logo') ? Storage::disk('public')->put('settings', $request->file('logo')) : $setting->logo;
            }

            if($setting->favicon){
                $favicon = $request->file('favicon') ? Storage::disk('public')->putFileAs(NULL, $request->file('favicon'), $setting->favicon) : $setting->favicon;
            }
            else{
                $favicon = $request->file('favicon') ? Storage::disk('public')->put('settings', $request->file('favicon')) : $setting->favicon;
            }

            $setting->update([
                'site_name' => $request->site_name ?? $setting->site_name,
                'description' => $request->description ?? $setting->description,
                'currency' => $request->currency ?? $setting->currency,
                's_currency' => $request->s_currency ?? $setting->s_currency,
                'bank_name' => $request->bank_name ?? $setting->bank_name,
                'account_name' => $request->account_name ?? $setting->account_name,
                'account_number' => $request->account_number ?? $setting->account_number,
                'eth_address' => $request->eth_address ?? $setting->eth_address,
                'btc_address' => $request->btc_address ?? $setting->btc_address,
                'ltc_address' => $request->ltc_address ?? $setting->ltc_address,
                'dodge_address' => $request->dodge_address ?? $setting->dodge_address,
                'bnb_address' => $request->bnb_address ?? $setting->bnb_address,
                'payment_mode' => $request->payment_mode ?? $setting->payment_mode,
                'location' => $request->location ?? $setting->location,
                's_s_k' => $request->s_s_k ?? $setting->s_s_k,
                's_p_k' => $request->s_p_k ?? $setting->s_p_k,
                'pp_cs' => $request->pp_cs ?? $setting->pp_cs,
                'pp_ci' => $request->pp_ci ?? $setting->pp_ci,
                'keywords' => $request->keywords ?? $setting->keywords,
                'site_title' => $request->site_title ?? $setting->site_title,
                'site_address' => $request->site_address ?? $setting->site_address,
                'logo' => $logo,
                'favicon' => $favicon,
                'trade_mode' => $request->trade_mode ?? $setting->trade_mode,
                'google_translate' => $request->google_translate ?? $setting->google_translate,
                'weekend_trade' => $request->weekend_trade ?? $setting->weekend_trade,
                'contact_email' => $request->contact_email ?? $setting->contact_email,
                'referral_commission' => $request->referral_commission ?? $setting->referral_commission,
                'referral_commission1' => $request->referral_commission1 ?? $setting->referral_commission1,
                'referral_commission2' => $request->referral_commission2 ?? $setting->referral_commission2,
                'referral_commission3' => $request->referral_commission3 ?? $setting->referral_commission3,
                'referral_commission4' => $request->referral_commission4 ?? $setting->referral_commission4,
                'referral_commission5' => $request->referral_commission5 ?? $setting->referral_commission5,
                'signup_bonus' => $request->signup_bonus ?? $setting->signup_bonus,
                'exchange_fee' => $request->exchange_fee ?? $setting->exchange_fee,
                'minimum_withdrawal' => $request->minimum_withdrawal ?? $setting->minimum_withdrawal,
                'maximum_withdrawal' => $request->maximum_withdrawal ?? $setting->maximum_withdrawal,
                'withdrawal_charge_percent' => $request->withdrawal_charge_percent ?? $setting->withdrawal_charge_percent,
                'files_key' => $request->files_key ?? $setting->files_key,
                'tawk_to' => $request->tawk_to ?? $setting->tawk_to,
                'enable_2fa' => $request->enable_2fa ? 'yes' : 'no',
                'enable_kyc' => $request->enable_kyc ? 'yes' : 'no',
                'enable_with' => $request->enable_with ? 'yes' : 'no',
                'enable_verification' => $request->enable_verification ? 'yes' : 'no',
                'withdrawal_option' => $request->withdrawal_option ?? $setting->withdrawal_option,
                'dashboard_option' => $request->dashboard_option ?? $setting->dashboard_option,
                'site_preference' => $request->site_preference ?? $setting->site_preference,
                'enable_annoc' => $request->enable_annoc ?? $setting->enable_annoc,
                'captcha' => $request->captcha ?? $setting->captcha,
                'commission_type' => $request->commission_type ?? $setting->commission_type,
                'commission_fee' => $request->commission_fee ?? $setting->commission_fee,
                'monthlyfee' => $request->monthlyfee ?? $setting->monthlyfee,
                'quarterlyfee' => $request->quarterlyfee ?? $setting->quarterlyfee,
                'yearlyfee' => $request->yearlyfee ?? $setting->yearlyfee,
                'newupdate' => $request->newupdate ?? $setting->newupdate,
                'updated_by' => $request->updated_by ?? $setting->updated_by,
            ]);

            toast("success", "settings updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }
}
