<?php

namespace App\Http\Controllers;

use Illuminate\Support\Facades\Storage;
use Illuminate\Http\Request;
use App\Models\Admin;
use App\Models\Plan;
use App\Models\User;
use App\Models\UserPlan;
use App\Models\Deposit;
use App\Models\Withdrawal;
use App\Models\TpTransaction;
use App\Models\Transfer;
use App\Models\Testimony;
use App\Models\Agent;
use App\Models\Faq;
use App\Models\Content;
use App\Models\Image;
use App\Models\Setting;
use App\Models\Notification;
use Carbon\Carbon;
use Mail;

class UserController extends Controller
{
    //
    public function __construct()
    {
    	$this->path = 'user/';
    }

    //
    public function register(Request $request)
    {
    	return view($this->path . 'register')->with([
    		'ref_by' => $request->ref_by
    	]);
    }

    //
    public function register_post(Request $request)
    {
    	$request->validate([
    		'name' => 'required',
    		'email' => 'required|unique:users,email',
    		'phone' => 'required',
    		'country' => 'required',
    		'password' => 'required|same:confirm_password',
    	]);

        try{

        	$referral = User::where("id", $request->ref_by)->first();

        	$user = User::create([
        		'name' => $request->name,
        		'email' => $request->email,
        		'phone' => $request->phone,
        		'country' => $request->country,
        		'ref_by' => $referral ? $request->ref_by : NULL,
        		'password' => bcrypt($request->password),
        	]);

        	//send verification mail if it was enabled by admin

            $data = array(
                'name' => $request->name
            );

            $mail = $request->email;
            
            Mail::send('mails.welcome', $data, function($message) use ($mail) {
                $message->to($mail)->subject('Welcome Mail');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            //notify the user
            $message = "With deep appreciation, we the team of " . app_name('separate') . " wish to welcome you to our platform. If you have any question or confusion, do not hesitate to contact our support for assistant!";
            notify($message, $user->id);

            //if email is enabled;
            if(enable_verification() == 'yes'){

                $token = generate_random_string(100);
                $user->two_factor_secret = $token;
                $user->save();

                $data = array(
                    'name' => $user->name,
                    'url' => route('user.verify', $token)
                );

                $mail = $user->email;
                
                Mail::send('mails.verify', $data, function($message) use ($mail) {
                    $message->to($mail)->subject('Verify Email');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                toast("info", "verify your email");
                return redirect()->back();
            }

            toast("success", "signup successful");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    //
    public function login()
    {
    	return view($this->path . 'login');
    }

    //
    public function login_post(Request $request)
    {

        try{
            $user = User::where("email", $request->email)->first();

            if(!$user){
                $body = 'email or username does not exist';
                toast('error', $body);
                return redirect()->back()->withInput();
            }

            if($user->status == 'blocked'){
                $body = 'your account has been blocked, please contact support for assistance';
                toast('error', $body);
                return redirect()->back()->withInput();
            }

            if(!\Hash::check($request->password, $user->password)){
                $body = 'Incorrect password, please try again';
                toast('error', $body);
                return redirect()->back()->withInput();
            }

            //if email is enabled;
            if(enable_verification() == 'yes' && $user->cstatus == NULL){

                $token = generate_random_string(100);
                $user->two_factor_secret = $token;
                $user->save();

                $data = array(
                    'name' => $user->name,
                    'url' => route('user.verify', $token)
                );

                $mail = $user->email;
                
                Mail::send('mails.verify', $data, function($message) use ($mail) {
                    $message->to($mail)->subject('Verify Email');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                toast("info", "verify your email");
                return redirect()->back();
            }

            //if 2fa is enabled;
            if(enable_2fa() == 'yes'){

                $token = generate_random_string(10);
                $user->two_factor_recovery_codes = $token;
                $user->save();

                $data = array(
                    'verify_token' => $token,
                );

                $mail = $user->email;
                
                Mail::send('mails.verify_2fa', $data, function($message) use ($mail) {
                    $message->to($mail)->subject('Verify Email');
                    $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
                });

                toast("info", "verify your email");
                return redirect()->route('user.verify_2fa');
            }

            \Auth::guard('web')->login($user);

            toast("success", "signin successful");
            return redirect()->route('user.dashboard');
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    //
    public function forgot()
    {
    	return view($this->path . 'forgot');
    }

    //
    public function forgot_post(Request $request)
    {
        $request->validate([
            'email' => 'required|exists:users,email'
        ]);

        try{
            $user = User::where("email", $request->email)->first();

            $token = generate_random_string(100);

            $user->two_factor_secret = $token;
            $user->save();

            $data = array(
                'name' => $user->name,
                'url' => route('user.reset_password', $token)
            );

            $mail = $user->email;
            
            Mail::send('mails.forgot', $data, function($message) use ($mail) {
                $message->to($mail)->subject('Reset Password');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            //notify the account holder about the reset
            $message = "A reset password process has been initiated with your account, if its not you, please ignore.";
            notify($message, $user->id);

            toast("success", "reset link sent successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back()->withInput();
        }
    }

    //
    public function reset_password(Request $request)
    {
    	return view($this->path . 'reset_password')->with([
    		'token' => $request->token
    	]);
    }

    //
    public function reset_password_post(Request $request)
    {
    	$request->validate([
    		'token' => 'required|exists:users,two_factor_secret',
    		'password' => 'required|same:confirm_password',
    	]);

        try{

	        $user = User::where("two_factor_secret", $request->token)->first();

            $token = generate_random_string(100);
            $user->two_factor_secret = $token;
            $user->save();

	        //update user record
	        $user->update([
	        	'password' => bcrypt($request->password),
	        ]);

            \Auth::guard('web')->login($user);

            toast("success", "password updated successfully");
            return redirect()->route('user.dashboard');
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function verify_2fa()
    {
        return view($this->path . 'verify_2fa');
    }

    //
    public function verify_2fa_post(Request $request)
    {
        $request->validate([
            'token' => 'required|exists:users,two_factor_recovery_codes',
        ]);

        try{

            $user = User::where("two_factor_recovery_codes", $request->token)->first();

            $token = generate_random_string(10);
            $user->two_factor_recovery_codes = $token;
            $user->save();

            \Auth::guard('web')->login($user);

            toast("success", "verification successful");
            return redirect()->route('user.dashboard');
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function verify(Request $request)
    {

        try{

            $user = User::where("two_factor_secret", $request->token)->first();

            if(!$user){
                toast("error", "token has expired");
                return redirect()->route('user.login');
            }

            $token = generate_random_string(100);
            $user->two_factor_secret = $token;
            $user->cstatus = 'yes';
            $user->save();

            \Auth::guard('web')->login($user);

            toast("success", "verification successful");
            return redirect()->route('user.dashboard');
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->route('user.login');
        }
    }

    //
    public function logout()
    {
        \Auth::guard('web')->logout();
        return redirect()->route('user.login');
    }

    //
    public function dashboard()
    {
        return view($this->path . 'dashboard');
    }

    //
    public function investment_plans()
    {
        return view($this->path . 'investment_plans')->with([
        	'plans' => Plan::all()
        ]);
    }

    //
    public function investment_plan_join(Request $request)
    {
        try{
        	$plan = Plan::where("id", $request->plan_id)->first();

        	if(!$plan){
	            toast("error", "plan not found");
	            return redirect()->back();
        	}

        	if($request->amount < $plan->min_price){
	            toast("error", "the amount you entered is lower than the minimum of this plan");
	            return redirect()->back();
        	}

        	if($request->amount > $plan->max_price){
	            toast("error", "the amount you entered is bigger than the maximum of this plan");
	            return redirect()->back();
        	}

        	if($request->amount > user()->account_bal){
	            toast("error", "your account balance is insufficient, please make a deposit");
	            return redirect()->route('user.deposit');
        	}

        	//deduct from user account balance
        	$user = user();
        	$user->update([
        		'account_bal' => ($user->account_bal - $request->amount)
        	]);

        	//join plan
        	UserPlan::create([
        		'plan' => $plan->id,
        		'user' => $user->id,
        		'amount' => $request->amount,
        		'active' => 'yes',
        		'tinterval' => $plan->increment_interval,
        		'shares' => $plan->increment_amount,
        		'inv_duration' => $plan->expiration,
        		'expire_date' => Carbon::now()->addHours(get_hours($plan->expiration)),
        		'activated_at' => Carbon::now(),
        		'last_growth' => Carbon::now(),
        	]);

        	//add to transaction
        	TpTransaction::create([
        		'user' => $user->id,
        		'plan' => $plan->name,
        		'amount' => $request->amount,
        		'type' => 'Plan Purchase',
        	]);

            //notify the user abou the investment
            $message = "You have successfully joined " . $plan->name . " Plan with a sum of $" . number_format($request->amount, 2, '.', ',');
            notify($message);

            toast("success", "you have joined the selected plan successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function my_packages()
    {
    	$plans = UserPlan::leftJoin("plans", "plans.id", "=", "user_plans.plan")
    					->select([
    						'plans.name as name',
    						'user_plans.id as id',
    						'user_plans.last_growth as last_growth',
    						'user_plans.expire_date as expire_date',
    						'user_plans.amount as amount',
    					])
    					->orderBy("user_plans.id", "desc")
    					->where("user_plans.user", user()->id)
    					->where("user_plans.active", "yes")
    					->get();

    	if(count($plans) > 0){
	        return view($this->path . 'my_packages')->with([
	        	'plans' => $plans
	        ]);
    	}
    	else{
            toast("error", "you do not have any active plan");
            return redirect()->back();
    	}
    }

    //
    public function deposit(Request $request)
    {
        return view($this->path . 'deposit')->with([
        	'deposits' => Deposit::orderBy("id", "desc")->where("user", user()->id)->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function payment(Request $request)
    {
        return view($this->path . 'payment')->with([
        	'setting' => Setting::first(),
        	'amount' => $request->amount
        ]);
    }

    //
    public function payment_submit(Request $request)
    {
        $request->validate([
            'amount' => 'required',
            'proof' => 'required|max:1024|mimes:jpg,jpeg,png',
        ]);

        try{

	        $previous = Deposit::orderBy("id", "desc")->where("user", user()->id)->where("status", "Pending")->first();

	        if($previous){
	            toast("error", "you have a pending payment, please wait for it to be confirmed before submitting another one");
	            return redirect()->back();
	        }

            $proof = Storage::disk("public")->put("proofs", $request->file("proof"));

            Deposit::create([
                'user' => user()->id,
                'amount' => $request->amount,
                'payment_mode' => $request->payment_mode,
                'status' => 'Pending',
                'proof' => $proof,
            ]);

            //notify the admin that someone has paid

            $data = array(
                'name' => user()->name,
                'amount' => $request->amount,
            );

            $mail = admin_email();
            
            Mail::send('mails.notify', $data, function($message) use ($mail) {
                $message->to($mail)->subject('Deposit Notification');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            toast("success", "payment submited successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function withdrawal(Request $request)
    {
        return view($this->path . 'withdrawal')->with([
        	'withdrawals' => Withdrawal::orderBy("id", "desc")->where("user", user()->id)->where("status", "Pending")->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function withdrawal_submit(Request $request)
    {
        $request->validate([
            'amount' => 'required',
        ]);

        try{
	        $user = user();

	        if($request->amount < minimum_withdrawal()){
	            toast("error", "your withdrawal amount is less than the minimum");
	            return redirect()->back();
	        }

	        if($request->amount > $user->account_bal){
	            toast("error", "insufficient balance");
	            return redirect()->back();
	        }

	        //check if the user have updated the address of the payment method
	        if($user[get_payment_mode_key($request->payment_mode)] == NULL){
	            toast("error", "please update your " . $request->payment_mode . " address before you chose it as payment mode");
	            return redirect()->route('user.withdrawal_details');
	        }

	        //deduct from user balance
	        $user->update([
	        	'account_bal' => ($user->account_bal - $request->amount)
	        ]);

            Withdrawal::create([
                'user' => user()->id,
                'amount' => $request->amount,
                'to_deduct' => withdrawal_charge_amount($request->amount),
                'payment_mode' => $request->payment_mode,
                'status' => 'Pending',
            ]);

            toast("success", "withdrawal submited successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function transfer(Request $request)
    {
        return view($this->path . 'transfer')->with([
            'transfers' => Transfer::leftJoin("users", "users.id", "=", "transfers.receiver")
                                ->select([
                                    'users.name as name',
                                    'users.email as email',
                                    'transfers.id as id',
                                    'transfers.amount as amount',
                                    'transfers.created_at as created_at',
                                ])
                                ->orderBy("transfers.id", "desc")
                                ->where("transfers.user", user()->id)
                                ->paginate(10),
            'search' => $request->search
        ]);
    }

    //
    public function transfer_post(Request $request)
    {
        $request->validate([
            'email' => 'required|exists:users,email',
            'amount' => 'required',
        ]);

        try{

            if($request->email == user()->email){
                toast("error", "you cannot transfer money to yourself");
                return redirect()->back();
            }

            $user = User::where("email", $request->email)->first();

            if($request->amount > user()->account_bal){
                toast("error", "insufficient balance");
                return redirect()->back();
            }

            //deduct from source balance
            user()->update([
                'account_bal' => (user()->account_bal - $request->amount)
            ]);

            //increase destination balance
            $user->update([
                'account_bal' => ($user->account_bal + $request->amount)
            ]);

            //add transaction for the source
            Transfer::create([
                'user' => user()->id,
                'receiver' => $user->id,
                'amount' => $request->amount,
            ]);

            //add transaction for the source
            TpTransaction::create([
                'user' => user()->id,
                'amount' => ($request->amount * -1),
                'type' => 'Transfer'
            ]);

            //add transaction for the destination
            TpTransaction::create([
                'user' => $user->id,
                'amount' => $request->amount,
                'type' => 'Transfer'
            ]);

            toast("success", "transfer made successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function withdrawal_details(Request $request)
    {
        return view($this->path . 'withdrawal_details');
    }

    //
    public function withdrawal_details_update(Request $request)
    {

        try{
	        $user = user();

	        //update user record
	        $user->update([
	        	'btc_address' => $request->btc_address ?? $user->btc_address,
	        	'eth_address' => $request->eth_address ?? $user->eth_address,
	        	'ltc_address' => $request->ltc_address ?? $user->ltc_address,
	        	'dodge_address' => $request->dodge_address ?? $user->dodge_address,
	        	'bnb_address' => $request->bnb_address ?? $user->bnb_address,
	        ]);

            toast("success", "withdrawal details updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function referral(Request $request)
    {
        return view($this->path . 'referral')->with([
        	'referrals' => User::where("ref_by", user()->id)->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function roi_history(Request $request)
    {
        return view($this->path . 'roi_history')->with([
        	'rois' => TpTransaction::where("user", user()->id)->where("type", "ROI")->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function deposit_history(Request $request)
    {
        return view($this->path . 'deposit_history')->with([
        	'deposits' => TpTransaction::where("user", user()->id)->where("type", "Deposit")->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function withdrawal_history(Request $request)
    {
        return view($this->path . 'withdrawal_history')->with([
        	'withdrawals' => TpTransaction::where("user", user()->id)->where("type", "Withdrawal")->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function other_history(Request $request)
    {
        return view($this->path . 'other_history')->with([
        	'others' => TpTransaction::where("user", user()->id)->whereNotIn("type", ["ROI", "Deposit", "Withdrawal"])->paginate(10),
        	'search' => $request->search
        ]);
    }

    //
    public function profile(Request $request)
    {
        return view($this->path . 'profile');
    }

    //
    public function profile_update(Request $request)
    {

        try{
	        $user = user();

	        //update user record
	        $user->update([
	        	'name' => $request->name ?? $user->name,
	        	'dob' => $request->dob ? Carbon::parse($request->dob) : $user->dob,
	        	'phone' => $request->phone ?? $user->phone,
	        	'address' => $request->address ?? $user->address,
	        ]);

            toast("success", "profile updated successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function change_password(Request $request)
    {

        try{
            $user = User::where("id", user()->id)->first();

            if(\Hash::check($request->old_password, $user->password)){
                if($request->new_password != $request->confirm_new_password){
                    toast("error", "password mismatch");
                    return redirect()->back();
                }

                $user->update([
                    'password' => $request->new_password ? bcrypt($request->new_password) : $user->password
                ]);

                toast("success", "profile updated successfully");
                return redirect()->back();
            }

            toast("error", "wrong old password");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function verify_account(Request $request)
    {
        return view($this->path . 'verify_account');
    }

    //
    public function verify_account_submit(Request $request)
    {
        $request->validate([
            'id_card' => 'required|max:1024|mimes:jpg,jpeg,png',
            'passport' => 'required|max:1024|mimes:jpg,jpeg,png',
        ]);

        try{

        	$user = user();

	        if($user->account_verify == "Verified"){
	            toast("info", "your account has already been verified");
	            return redirect()->back();
	        }

            if($user->id_card){
                $id_card = $request->file('id_card') ? Storage::disk('public')->putFileAs(NULL, $request->file('id_card'), $user->id_card) : $user->id_card;
            }
            else{
                $id_card = $request->file('id_card') ? Storage::disk('public')->put('idcards', $request->file('id_card')) : $user->id_card;
            }

            if($user->passport){
                $passport = $request->file('passport') ? Storage::disk('public')->putFileAs(NULL, $request->file('passport'), $user->passport) : $user->passport;
            }
            else{
                $passport = $request->file('passport') ? Storage::disk('public')->put('passports', $request->file('passport')) : $user->passport;
            }

            $user->update([
                'id_card' => $id_card,
                'passport' => $passport,
                'account_verify' => 'Under review',
            ]);

            //notify the admin that someone has uploaded kyc

            $data = array(
                'name' => user()->name,
            );

            $mail = admin_email();
            
            Mail::send('mails.notify4', $data, function($message) use ($mail) {
                $message->to($mail)->subject('KYC Upload Notification');
                $message->from(env('MAIL_FROM_ADDRESS'), env('MAIL_FROM_NAME'));
            });

            toast("success", "KYC documents submited successfully");
            return redirect()->back();
        }
        catch(\Throwable $e){
            toast("error", $e->getMessage());
            return redirect()->back();
        }
    }

    //
    public function notification(Request $request)
    {
        if($request->new){
            $notifications = Notification::orderBy("id", "desc")->where("user_id", user()->id)->where("has_seen", NULL)->where("message", "like", "%" . $request->search . "%")->paginate(10);
            Notification::where("user_id", user()->id)->where("has_seen", NULL)->update([
                'has_seen' => 'yes'
            ]);
            $new = 'yes';
        }
        else{
            $notifications = Notification::orderBy("id", "desc")->where("message", "like", "%" . $request->search . "%")->paginate(10);
            $new = NULL;
        }

        return view($this->path . 'notification')->with([
            'notifications' => $notifications,
            'new' => $new,
            'search' => $request->search
        ]);
    }

    //
    public function support(Request $request)
    {
        return view($this->path . 'support');
    }
}
